/*!
 * Log.js
 * Copyright(c) 2010 TJ Holowaychuk <tj@vision-media.ca>
 * MIT Licensed
 */

/**
 * Module dependencies.
 */

var colors = require('colors');
var fmt = require('util').format;
var EventEmitter = require('events').EventEmitter;

/**
 * Initialize a `Loggeer` with the given log `level` defaulting
 * to __DEBUG__ and `stream` defaulting to _stdout_.
 *
 * @param {Number} level
 * @param {Object} stream
 * @api public
 */

var Log = exports = module.exports = function Log(level, stream){
  if ('string' == typeof level) level = exports[level.toUpperCase()];
  this.level = isFinite(level) ? level : this.DEBUG;
  if(!process.browser) {
    this.stream = stream || process.stdout;
    if (this.stream.readable) this.read();
  }
};

/**
 * System is unusable.
 *
 * @type Number
 */

exports.EMERGENCY = 0;

/**
 * Action must be taken immediately.
 *
 * @type Number
 */

exports.ALERT = 1;

/**
 * Critical condition.
 *
 * @type Number
 */

exports.CRITICAL = 2;

/**
 * Error condition.
 *
 * @type Number
 */

exports.ERROR = 3;

/**
 * Warning condition.
 *
 * @type Number
 */

exports.WARNING = 4;

/**
 * Normal but significant condition.
 *
 * @type Number
 */

exports.NOTICE = 5;

/**
 * Purely informational message.
 *
 * @type Number
 */

exports.INFO = 6;

/**
 * Application debug messages.
 *
 * @type Number
 */

exports.DEBUG = 7;

/**
 * prototype.
 */

Log.prototype = {

  /**
   * Start emitting "line" events.
   *
   * @api public
   */

  read: function(){
    var buf = ''
      , self = this
      , stream = this.stream;

    stream.setEncoding('utf8');
    stream.on('data', function(chunk){
      buf += chunk;
      if ('\n' != buf[buf.length - 1]) return;
      buf.split('\n').map(function(line){
        if (!line.length) return;
        try {
          var captures = line.match(/^\[([^\]]+)\] (\w+) (.*)/);
          var obj = {
              date: new Date(captures[1])
            , level: exports[captures[2]]
            , levelString: captures[2]
            , msg: captures[3]
          };
          self.emit('line', obj);
        } catch (err) {
          // Ignore
        }
      });
      buf = '';
    });

    stream.on('end', function(){
      self.emit('end');
    });
  },

  isEnable: function(levelStr) {
    return exports[levelStr] <= this.level;
  },

  getTimeStr: function() {
    var date = '[' + new Date + ']';
    if (this.stream === process.stdout) {
      return date.grey;
    }
    return date;
  },

  getLevelStr: function(levelStr) {
    if (this.stream === process.stdout) {
      var result;
      switch (levelStr) {
        case 'DEBUG':
          result = levelStr.cyan;
          break;
        case 'INFO':
          result = levelStr.green;
          break;
        case 'NOTICE':
          result = levelStr.blue;
          break;
        case 'WARNING':
          result = levelStr.yellow;
          break;
        case 'ERROR':
          result = levelStr.red;
          break;
        case 'CRITICAL':
          result = levelStr.red.bold;
          break;
        case 'ALERT':
          result = levelStr.magenta;
          break;
        case 'EMERGENCY':
          result = levelStr.black.bgRed;
          break;
        default:
          result = levelStr;
      }
      return result;
    }
    return levelStr;
  },

  /**
   * Log output message.
   *
   * @param  {String} levelStr
   * @param  {Array} args
   * @api private
   */

  log: function(levelStr, args) {
    if (this.isEnable(levelStr)) {
      var msg = fmt.apply(null, args);
      if (!process.browser) {
        this.stream.write(
          this.getTimeStr()
          + ' ' + this.getLevelStr(levelStr)
          + ' ' + msg
          + '\n'
        )
        console.log(this.getTimeStr()
          + ' ' + this.getLevelStr(levelStr)
          + ' ' + msg)
      } else {
        if (window.console) {
          console.log(this.getTimeStr()
          + ' ' + this.getLevelStr(levelStr)
          + ' ' + msg)
        }
      }
    }
  },

  /**
   * Log emergency `msg`.
   *
   * @param  {String} msg
   * @api public
   */

  emergency: function(msg){
    this.log('EMERGENCY', arguments);
  },

  /**
   * Log alert `msg`.
   *
   * @param  {String} msg
   * @api public
   */

  alert: function(msg){
    this.log('ALERT', arguments);
  },

  /**
   * Log critical `msg`.
   *
   * @param  {String} msg
   * @api public
   */

  critical: function(msg){
    this.log('CRITICAL', arguments);
  },

  /**
   * Log error `msg`.
   *
   * @param  {String} msg
   * @api public
   */

  error: function(msg){
    this.log('ERROR', arguments);
  },

  /**
   * Log warning `msg`.
   *
   * @param  {String} msg
   * @api public
   */

  warning: function(msg){
    this.log('WARNING', arguments);
  },

  /**
   * Log notice `msg`.
   *
   * @param  {String} msg
   * @api public
   */

  notice: function(msg){
    this.log('NOTICE', arguments);
  },

  /**
   * Log info `msg`.
   *
   * @param  {String} msg
   * @api public
   */

  info: function(msg){
    this.log('INFO', arguments);
  },

  /**
   * Log debug `msg`.
   *
   * @param  {String} msg
   * @api public
   */

  debug: function(msg){
    this.log('DEBUG', arguments);
  }
};

/**
 * Inherit from `EventEmitter`.
 */

Log.prototype.__proto__ = EventEmitter.prototype;
